# Reglas y Convenciones del Proyecto Ser-Ajayu

## Stack Tecnológico

- **Backend**: Laravel 12, PHP 8.3
- **Frontend**: React, TypeScript, Inertia.js v2
- **Estilos**: TailwindCSS + shadcn/ui
- **Testing**: Pest v4
- **Routing**: Wayfinder (generación automática de rutas TypeScript)
- **Permisos**: Spatie Laravel Permission

## Reglas de Comunicación

### Explicar Antes de Implementar

**Antes de implementar cualquier funcionalidad**, el agente debe:

1. **Explicar QUÉ va a hacer** y por qué es necesario
2. **Distinguir entre lo obligatorio y lo opcional/conveniente**
3. **Mencionar alternativas** si existen
4. **Pedir confirmación** si la implementación tiene múltiples partes

### Diferencias Inertia vs Livewire

| Aspecto           | Livewire                   | Inertia                    |
| ----------------- | -------------------------- | -------------------------- |
| Renderizado       | Server (Blade)             | Client (React)             |
| Acceso a PHP      | Directo (`@can`, `auth()`) | Vía props JSON             |
| Permisos en vista | `@can('permission')`       | Enviar como prop + hook JS |

## Reglas de Base de Datos

### Migraciones

- **NO crear migraciones `add_*`** para añadir campos a tablas existentes durante desarrollo
- **Modificar directamente** la migración original de la tabla
- Solo crear migraciones separadas cuando la app ya está en producción
- Después de modificar una migración, usar `php artisan migrate:fresh --seed`

### Rutas

El proyecto usa **4 archivos de rutas** separados por rol:

| Archivo              | Middleware       | Descripción                                 |
| -------------------- | ---------------- | ------------------------------------------- |
| `routes/web.php`     | -                | Rutas públicas, landing, páginas de lectura |
| `routes/user.php`    | `auth, verified` | Rutas de usuario autenticado                |
| `routes/curator.php` | `role:curator`   | Rutas de curador (moderación)               |
| `routes/admin.php`   | `role:admin`     | Panel de administración                     |

**Convenciones:**

- Nuevas rutas públicas → `web.php`
- Nuevas rutas que requieren login → `user.php`
- Rutas de moderación/curación → `curator.php`
- Rutas de gestión/admin → `admin.php`

---

## MCPs Disponibles

- **GitHub**: Control de versiones, PRs, issues
- **Context7**: Documentación actualizada - usar `/websites/laravel_12_x` para Laravel 12
- **Laravel Boost** (v1.8+): MCP local con herramientas especializadas para Laravel

### Laravel Boost Tools

| Tool               | Descripción                                                              |
| ------------------ | ------------------------------------------------------------------------ |
| `search-docs`      | Buscar documentación de Laravel, Inertia, Pest, Tailwind, Filament, etc. |
| `application-info` | Info de PHP, Laravel, paquetes instalados y modelos                      |
| `database-query`   | Ejecutar queries SQL read-only                                           |
| `database-schema`  | Ver schema de la base de datos                                           |
| `tinker`           | Ejecutar código PHP en contexto de la aplicación                         |
| `list-routes`      | Ver rutas de la aplicación                                               |
| `last-error`       | Ver último error de la aplicación                                        |
| `browser-logs`     | Leer logs del navegador (debugging frontend)                             |

> **Usar workflow `/search-docs`** para buscar documentación del ecosistema Laravel.

---

## Arquitectura: Por Funcionalidad

### Backend (PHP)

```
app/
├── Http/
│   ├── Controllers/{Funcionalidad}/{Nombre}Controller.php
│   └── Requests/{Funcionalidad}/{Accion}{Nombre}Request.php
├── Actions/{Funcionalidad}/{Verbo}{Nombre}Action.php
└── Services/{Funcionalidad}/{Nombre}Service.php
```

### Frontend (React/TypeScript)

```
resources/js/
├── pages/{Funcionalidad}/{Pagina}.tsx       # Mirror de Controllers
├── components/
│   ├── tables/                              # DataTable, TablePagination
│   ├── modals/                              # ConfirmModal, FormModal
│   ├── search/                              # SearchBar, SearchFilters
│   ├── forms/                               # Input, Select, Button
│   ├── navigation/                          # Sidebar, Navbar
│   ├── ui/                                  # Card, Badge, Tooltip (shadcn)
│   └── feedback/                            # Alert, Toast, Spinner
├── layouts/{Nombre}Layout.tsx
├── hooks/use{Nombre}.ts
└── types/
```

---

## Convenciones de Nomenclatura

### PHP

| Tipo          | Formato                   | Ejemplo                                 |
| ------------- | ------------------------- | --------------------------------------- |
| Controllers   | `{Nombre}Controller`      | `UserController`, `LoginController`     |
| Form Requests | `{Accion}{Nombre}Request` | `StoreUserRequest`, `UpdateUserRequest` |
| Actions       | `{Verbo}{Nombre}Action`   | `CreateUserAction`, `DeleteOrderAction` |
| Services      | `{Nombre}Service`         | `PaymentService`, `NotificationService` |
| Models        | Singular, PascalCase      | `User`, `Order`, `ProductCategory`      |

### React/TypeScript

| Tipo             | Formato                       | Ejemplo                                  |
| ---------------- | ----------------------------- | ---------------------------------------- |
| Componentes      | PascalCase (sin sufijo)       | `DataTable`, `SearchBar`, `ConfirmModal` |
| Páginas          | PascalCase, espejo del método | `Index`, `Show`, `Edit`, `Create`        |
| Hooks            | camelCase con prefijo `use`   | `useDebounce`, `useAuth`, `useSearch`    |
| Types/Interfaces | PascalCase                    | `User`, `PaginatedResponse<T>`           |
| Archivos         | kebab-case                    | `data-table.tsx`, `use-debounce.ts`      |

---

## Correspondencia Backend ↔ Frontend

| Backend                                   | Frontend                 |
| ----------------------------------------- | ------------------------ |
| `Controllers/Auth/LoginController.php`    | `pages/auth/login.tsx`   |
| `Controllers/Users/UserController@index`  | `pages/users/index.tsx`  |
| `Controllers/Users/UserController@edit`   | `pages/users/edit.tsx`   |
| `Controllers/Users/UserController@show`   | `pages/users/show.tsx`   |
| `Controllers/Users/UserController@create` | `pages/users/create.tsx` |

---

## Validaciones

### Prioridad: Form Request

**SIEMPRE** usar Form Request para validaciones. Esto asegura:

- Separación de responsabilidades
- Reutilización de reglas
- Mensajes de error consistentes
- Autorización integrada

```php
// ✅ Correcto: Form Request
public function store(StoreUserRequest $request): RedirectResponse
{
    // $request ya está validado
}

// ❌ Evitar: Validación inline
public function store(Request $request): RedirectResponse
{
    $request->validate([...]); // Solo en casos muy específicos
}
```

### Excepciones (validación en Controller)

Solo validar en controller cuando:

- La lógica depende de servicios externos
- Hay validaciones asíncronas complejas
- El contexto de validación cambia dinámicamente

---

## Wayfinder

Wayfinder genera funciones TypeScript automáticamente desde las rutas de Laravel.

```typescript
// ✅ Correcto: Import nombrado (tree-shakable)
import { store, index } from '@/actions/App/Http/Controllers/UserController';

// Con Inertia useForm
const form = useForm({ name: '' });
form.submit(store());

// ❌ Evitar: Import default
import UserController from '@/actions/...';
```

Después de modificar rutas, ejecutar:

```bash
php artisan wayfinder:generate
```

---

## Testing

### Ubicación de Tests

- **Feature Tests**: `tests/Feature/{Funcionalidad}/`
- **Unit Tests**: `tests/Unit/{Funcionalidad}/`
- **Browser Tests**: `tests/Browser/` (Pest 4)

### Comandos

```bash
# Test específico
php artisan test --compact --filter=testName

# Test de archivo
php artisan test --compact tests/Feature/Users/UserTest.php

# Todos los tests
php artisan test --compact
```

---

## Flujo de Trabajo Recomendado

1. **Crear módulo**: Usar workflow `/create-module`
2. **Agregar validaciones**: Crear Form Request con `/create-form-request`
3. **Lógica de negocio**: Extraer a Action con `/create-action`
4. **Componentes reutilizables**: Usar `/create-component`
5. **Testing**: Ejecutar tests con `/run-tests`

---

## Comandos Artisan Frecuentes

```bash
# Crear controller con recursos
php artisan make:controller {Funcionalidad}/{Nombre}Controller --resource --no-interaction

# Crear Form Request
php artisan make:request {Funcionalidad}/{Accion}{Nombre}Request --no-interaction

# Crear modelo con todo
php artisan make:model {Nombre} -mfsc --no-interaction

# Formatear código
vendor/bin/pint --dirty
```

---

## Notas Importantes

1. **No crear archivos de documentación** sin aprobación explícita
2. **Siempre correr tests** después de cambios
3. **Usar `--no-interaction`** en comandos artisan
4. **Formatear con Pint** antes de finalizar cambios
5. **Generar Wayfinder** después de modificar rutas
