---
description: Ejecutar tests PHP (Pest) y frontend
---

# Ejecutar Tests

// turbo-all

Este workflow ejecuta tests del proyecto.

## Tests PHP (Pest)

### Ejecutar todos los tests

```bash
php artisan test --compact
```

### Ejecutar tests de un archivo específico

```bash
php artisan test --compact tests/Feature/{Funcionalidad}/{Nombre}Test.php
```

### Filtrar por nombre de test

```bash
php artisan test --compact --filter="nombre del test"
```

### Ejecutar solo tests unitarios

```bash
php artisan test --compact tests/Unit
```

### Ejecutar solo tests feature

```bash
php artisan test --compact tests/Feature
```

### Ejecutar tests con coverage

```bash
php artisan test --coverage
```

### Ejecutar tests con coverage mínimo

```bash
php artisan test --coverage --min=80
```

## Tests de Browser (Pest 4)

### Ejecutar tests de browser

```bash
php artisan test --compact tests/Browser
```

### Test de browser específico

```bash
php artisan test --compact tests/Browser/{Nombre}Test.php
```

## Verificación de Código

### Formatear con Pint (solo archivos modificados)

```bash
vendor/bin/pint --dirty
```

### Formatear todo el código

```bash
vendor/bin/pint
```

### Verificar sin cambiar (CI/CD)

```bash
vendor/bin/pint --test
```

## Tests Frontend

### Ejecutar tests de TypeScript/React (si están configurados)

```bash
npm run test
```

### Verificar tipos TypeScript

```bash
npm run type-check
```

### Verificar linting

```bash
npm run lint
```

## Flujo Recomendado

1. **Durante desarrollo**: Ejecutar tests específicos frecuentemente

    ```bash
    php artisan test --compact --filter="testName"
    ```

2. **Antes de commit**: Formatear y ejecutar tests relacionados

    ```bash
    vendor/bin/pint --dirty
    php artisan test --compact tests/Feature/{Funcionalidad}/
    ```

3. **Antes de PR**: Ejecutar suite completa
    ```bash
    vendor/bin/pint
    php artisan test --compact
    ```

## Crear Nuevos Tests

### Test Feature

```bash
php artisan make:test {Funcionalidad}/{Nombre}Test --pest --no-interaction
```

### Test Unit

```bash
php artisan make:test {Funcionalidad}/{Nombre}Test --pest --unit --no-interaction
```

## Ejemplo de Test Pest

```php
<?php

use App\Models\User;

it('can list users', function () {
    $users = User::factory()->count(3)->create();

    $response = $this->actingAs(User::factory()->create())
        ->get(route('users.index'));

    $response->assertOk()
        ->assertInertia(fn ($page) => $page
            ->component('Users/Index')
            ->has('users', 3)
        );
});

it('can create a user', function () {
    $admin = User::factory()->admin()->create();

    $response = $this->actingAs($admin)
        ->post(route('users.store'), [
            'name' => 'John Doe',
            'email' => 'john@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ]);

    $response->assertRedirect();

    expect(User::where('email', 'john@example.com')->exists())->toBeTrue();
});

it('validates required fields', function () {
    $admin = User::factory()->admin()->create();

    $response = $this->actingAs($admin)
        ->post(route('users.store'), []);

    $response->assertSessionHasErrors(['name', 'email', 'password']);
});
```

## Solución de Problemas

### Tests fallan por base de datos

```bash
php artisan migrate:fresh --seed --env=testing
```

### Tests fallan por caché

```bash
php artisan config:clear
php artisan cache:clear
php artisan route:clear
```

### Regenerar autoload

```bash
composer dump-autoload
```
