<?php

namespace App\Concerns;

use App\Models\User;
use Illuminate\Validation\Rule;

trait ProfileValidationRules
{
    /**
     * Get the validation rules used to validate user profiles.
     *
     * @return array<string, array<int, \Illuminate\Contracts\Validation\Rule|array<mixed>|string>>
     */
    protected function profileRules(?int $userId = null): array
    {
        return [
            'name' => $this->nameRules(),
            'email' => $this->emailRules($userId),
            'age' => ['nullable', 'integer', 'min:18', 'max:120'],
            'gender' => ['nullable', 'string', 'in:male,female,other,prefer_not_to_say'],
            'country' => ['nullable', 'string', 'max:100'],
            'entheogen_activity' => ['nullable', 'string', 'max:1000'],
        ];
    }

    /**
     * Get the validation rules for registration (includes extra profile fields).
     *
     * @return array<string, array<int, \Illuminate\Contracts\Validation\Rule|array<mixed>|string>>
     */
    protected function registrationProfileRules(): array
    {
        return [
            'name' => $this->nameRules(),
            'email' => $this->emailRules(),
            'age' => ['nullable', 'integer', 'min:18', 'max:120'],
            'gender' => ['nullable', 'string', 'in:male,female,other,prefer_not_to_say'],
            'country' => ['nullable', 'string', 'max:100'],
            'entheogen_activity' => ['nullable', 'string', 'max:1000'],
            'avatar' => [
                'nullable',
                'image',
                'mimes:jpg,jpeg,png,webp',
                'max:2048', // 2MB max
                'dimensions:min_width=100,min_height=100,max_width=2000,max_height=2000',
            ],
        ];
    }

    /**
     * Get the validation rules used to validate user names.
     *
     * @return array<int, \Illuminate\Contracts\Validation\Rule|array<mixed>|string>
     */
    protected function nameRules(): array
    {
        return ['required', 'string', 'max:255'];
    }

    /**
     * Get the validation rules used to validate user emails.
     *
     * @return array<int, \Illuminate\Contracts\Validation\Rule|array<mixed>|string>
     */
    protected function emailRules(?int $userId = null): array
    {
        return [
            'required',
            'string',
            'email',
            'max:255',
            $userId === null
                ? Rule::unique(User::class)
                : Rule::unique(User::class)->ignore($userId),
        ];
    }
}
