# Documentación de Implementación - Ser Ajayu

> **Última actualización:** 2026-01-20
> **Estado del proyecto:** En desarrollo activo
> **Versión:** 0.4.0

---

## 📋 Prompt de Contexto para IA

```
Eres un asistente de desarrollo trabajando en el proyecto "Ser Ajayu", una plataforma web para la integración de saberes ancestrales andinos/amazónicos con investigación científica moderna sobre enteógenos.

ANTES DE HACER CUALQUIER COSA:

1. LEE los archivos de reglas y convenciones:
   - `.agent/rules.md` - Reglas generales del proyecto
   - `GEMINI.md` - Guidelines de Laravel Boost (en la raíz)

2. LEE los workflows disponibles:
   - `.agent/workflows/` - Workflows para crear módulos, componentes, etc.

3. LEE la documentación del proyecto:
   - `docs/information.md` - Ficha técnica y objetivos
   - `docs/IMPLEMENTATION.md` - Este archivo (estado actual)
   - `docs/Pantallas.png` - Wireframes de las pantallas

4. USA Laravel Boost MCP para:
   - `search-docs` - Buscar documentación de Laravel, Inertia, Pest, etc.
   - `database-schema` - Ver estructura de tablas
   - `database-query` - Ejecutar queries read-only
   - `tinker` - Ejecutar código PHP en contexto de la app

5. EJECUTA estos comandos para entender el estado:
   - `php artisan route:list` - Ver rutas registradas
   - `php artisan permission:show` - Ver roles y permisos
   - `npm run dev` o `composer run dev` - Iniciar desarrollo

El proyecto usa:
- Laravel 12 + PHP 8.3
- React 19 + TypeScript + Inertia.js v2
- TailwindCSS v4 + shadcn/ui
- Spatie Laravel Permission
- Pest v4 para testing
- Wayfinder para rutas TypeScript

IMPORTANTE: Siempre explica lo que vas a hacer antes de implementar. Distingue entre lo obligatorio y lo opcional.
```

---

## 📊 Índice

1. [Stack Tecnológico](#-stack-tecnológico)
2. [Estado Actual del Proyecto](#-estado-actual-del-proyecto)
3. [Funcionalidades Implementadas](#-funcionalidades-implementadas)
4. [Estructura de Archivos](#-estructura-de-archivos)
5. [Sistema de Autenticación](#-sistema-de-autenticación)
6. [Sistema de Roles y Permisos](#-sistema-de-roles-y-permisos)
7. [Módulo de Foro](#-módulo-de-foro)
8. [Sistema de Notificaciones](#-sistema-de-notificaciones)
9. [Rutas de la Aplicación](#-rutas-de-la-aplicación)
10. [Base de Datos](#-base-de-datos)
11. [Componentes UI Clave](#-componentes-ui-clave)
12. [Pendientes y Próximos Pasos](#-pendientes-y-próximos-pasos)
13. [Problemas Conocidos y Soluciones](#-problemas-conocidos-y-soluciones)

---

## 🛠 Stack Tecnológico

| Tecnología        | Versión | Uso                               |
| ----------------- | ------- | --------------------------------- |
| Laravel           | 12      | Backend, API, Autenticación       |
| PHP               | 8.3.6   | Lenguaje servidor                 |
| React             | 19      | Frontend SPA                      |
| TypeScript        | -       | Tipado frontend                   |
| Inertia.js        | v2      | Puente Laravel-React sin API REST |
| TailwindCSS       | v4      | Estilos utility-first             |
| shadcn/ui         | -       | Componentes UI base               |
| Spatie Permission | 6.24    | Roles y permisos                  |
| Laravel Fortify   | v1      | Autenticación backend             |
| Laravel Socialite | v5      | Login social (Google, etc.)       |
| Wayfinder         | v0      | Generación rutas TypeScript       |
| Pest              | v4      | Testing PHP                       |
| Framer Motion     | -       | Animaciones React                 |

---

## 📈 Estado Actual del Proyecto

### Resumen Ejecutivo

El proyecto tiene implementado:

- ✅ Sistema de autenticación completo (registro, login, 2FA, social login)
- ✅ Sistema de roles y permisos (admin, curator, user)
- ✅ Foro funcional con threads, posts, votaciones y moderación
- ✅ Sistema de notificaciones en tiempo real
- ✅ Perfil de usuario con foto, campos adicionales
- ✅ Landing page con diseño premium
- ✅ Layouts público y autenticado

### Pendiente de implementar:

- ✅ Repositorio de documentos (Integración visual con Google Drive + Galería)
- ⏳ Documentos colaborativos (editor en tiempo real)
- ⏳ Encuestas digitales
- ⏳ Mensajería privada/grupal
- ⏳ Panel de administración completo
- ⏳ Multilenguaje (ES, EN, Quechua)

---

## ✅ Funcionalidades Implementadas

### Autenticación y Usuarios

| Feature                          | Estado | Archivos Clave                                 |
| -------------------------------- | ------ | ---------------------------------------------- |
| Registro con campos de perfil    | ✅     | `CreateNewUser.php`, `register.tsx`            |
| Login con email/password         | ✅     | `login.tsx`, Fortify                           |
| Login con Google OAuth           | ✅     | `SocialAuthController.php`, `social-login.tsx` |
| Password con toggle visibilidad  | ✅     | `password-input.tsx`                           |
| Two-Factor Authentication        | ✅     | Fortify built-in                               |
| Recuperación de contraseña       | ✅     | `reset-password.tsx`, `forgot-password.tsx`    |
| Verificación de email            | ✅     | `verify-email.tsx`                             |
| Foto de perfil (upload/delete)   | ✅     | `ProfileController.php`, `profile.tsx`         |
| Campos adicionales (age, gender) | ✅     | `ProfileValidationRules.php`                   |
| Redirección post-login por rol   | ✅     | `LoginResponse.php`, `RegisterResponse.php`    |

### Foro de Discusión

| Feature                           | Estado | Archivos Clave                           |
| --------------------------------- | ------ | ---------------------------------------- |
| Listado de foros                  | ✅     | `ForumController.php`, `forum/index.tsx` |
| Ver foro con threads              | ✅     | `forum/show.tsx`                         |
| Crear threads                     | ✅     | `ForumThreadController.php`              |
| Ver thread con posts              | ✅     | `forum/thread/show.tsx`                  |
| Responder a threads               | ✅     | `PostController.php`                     |
| Votaciones (upvote/downvote)      | ✅     | `vote-control.tsx`                       |
| Moderación (sticky, lock, delete) | ✅     | `ForumModerationController.php`          |
| Acceso público (lectura)          | ✅     | Rutas en `user.php`                      |

### Sistema de Notificaciones

| Feature                    | Estado | Archivos Clave               |
| -------------------------- | ------ | ---------------------------- |
| Notificación de respuestas | ✅     | `PostReply.php`              |
| Notificación de upvotes    | ✅     | `PostUpvoted.php`            |
| Dropdown de notificaciones | ✅     | `notifications-dropdown.tsx` |
| Marcar como leída          | ✅     | `NotificationController.php` |
| Contador en navbar         | ✅     | `HandleInertiaRequests.php`  |

### UI/UX

| Feature                        | Estado | Archivos Clave          |
| ------------------------------ | ------ | ----------------------- |
| Landing page animada           | ✅     | `welcome.tsx`           |
| Public layout (navbar, footer) | ✅     | `public-layout.tsx`     |
| App layout (dashboard)         | ✅     | `app-layout.tsx`        |
| Bottom tab navigator           | ✅     | `bottom-nav.tsx`        |
| User dropdown (avatar, logout) | ✅     | `user-dropdown.tsx`     |
| Dark/Light mode toggle         | ✅     | `appearance-toggle.tsx` |
| Responsive design              | ✅     | TailwindCSS breakpoints |

---

## 📁 Estructura de Archivos

### Backend (PHP/Laravel)

```
app/
├── Actions/
│   └── Fortify/
│       ├── CreateNewUser.php           # Registro con avatar y campos
│       └── UpdateUserProfileInformation.php
├── Concerns/
│   └── ProfileValidationRules.php      # Reglas de validación perfil
├── Http/
│   ├── Controllers/
│   │   ├── Admin/
│   │   │   ├── DashboardController.php
│   │   │   └── UserController.php
│   │   ├── Auth/
│   │   │   └── SocialAuthController.php  # Google OAuth
│   │   ├── Forum/
│   │   │   ├── ForumController.php
│   │   │   ├── ForumModerationController.php
│   │   │   ├── ForumThreadController.php
│   │   │   └── PostController.php
│   │   ├── Settings/
│   │   │   ├── PasswordController.php
│   │   │   ├── ProfileController.php     # Avatar upload/delete
│   │   │   └── TwoFactorController.php
│   │   └── NotificationController.php
│   ├── Middleware/
│   │   ├── HandleAppearance.php
│   │   └── HandleInertiaRequests.php     # Props compartidos (auth, roles, notifications)
│   ├── Requests/
│   │   ├── Forum/
│   │   │   ├── StoreForumPostRequest.php
│   │   │   └── StoreForumThreadRequest.php
│   │   ├── Profile/
│   │   │   └── UpdateAvatarRequest.php
│   │   └── Settings/
│   └── Responses/
│       ├── LoginResponse.php             # Redirección por rol
│       └── RegisterResponse.php          # Redirección por rol
├── Models/
│   ├── Forum.php
│   ├── ForumPost.php
│   ├── ForumThread.php
│   ├── ForumVote.php
│   ├── SocialAccount.php
│   └── User.php                          # HasRoles, profile_photo_url
├── Notifications/
│   └── Forum/
│       ├── PostReply.php
│       └── PostUpvoted.php
└── Providers/
    └── FortifyServiceProvider.php        # Registra respuestas personalizadas

routes/
├── web.php                               # Rutas públicas
├── user.php                              # Usuario autenticado + Foro
├── curator.php                           # Curador (moderación)
├── admin.php                             # Administración
├── settings.php                          # Configuración perfil
└── console.php                           # Comandos

database/
├── factories/
│   ├── ForumFactory.php
│   ├── ForumPostFactory.php
│   ├── ForumThreadFactory.php
│   └── UserFactory.php
├── migrations/
│   ├── 0001_01_01_000000_create_users_table.php  # Con avatar, age, gender, etc.
│   ├── 2026_01_16_140531_create_permission_tables.php
│   ├── 2026_01_16_195320_create_forums_table.php
│   ├── 2026_01_16_195321_create_forum_threads_table.php
│   ├── 2026_01_16_195322_create_forum_posts_table.php
│   ├── 2026_01_16_195323_create_forum_votes_table.php
│   ├── 2026_01_16_215241_create_notifications_table.php
│   └── 2026_01_17_020818_create_social_accounts_table.php
└── seeders/
    ├── DatabaseSeeder.php
    ├── ForumSeeder.php
    └── RolesAndPermissionsSeeder.php
```

### Frontend (React/TypeScript)

```
resources/js/
├── actions/                              # Generado por Wayfinder
├── components/
│   ├── auth/
│   │   └── social-login.tsx              # Botones de login social
│   ├── forum/
│   │   ├── create-thread-form.tsx
│   │   ├── forum-post-item.tsx
│   │   ├── forum-reply-form.tsx
│   │   ├── forum-thread-item.tsx
│   │   └── vote-control.tsx
│   ├── navigation/
│   │   └── bottom-nav.tsx
│   ├── ui/
│   │   ├── avatar.tsx
│   │   ├── button.tsx
│   │   ├── password-input.tsx            # Input con toggle visibilidad
│   │   └── ... (shadcn components)
│   ├── notifications-dropdown.tsx
│   └── user-dropdown.tsx
├── hooks/
│   └── use-permissions.ts
├── layouts/
│   ├── app-layout.tsx                    # Layout dashboard
│   ├── auth-layout.tsx                   # Layout login/register
│   ├── public-layout.tsx                 # Layout público con navbar
│   └── settings/
│       └── layout.tsx
├── pages/
│   ├── admin/
│   │   ├── dashboard.tsx
│   │   ├── users/
│   │   │   ├── index.tsx
│   │   │   └── create.tsx
│   ├── auth/
│   │   ├── confirm-password.tsx
│   │   ├── forgot-password.tsx
│   │   ├── login.tsx
│   │   ├── register.tsx                  # Con avatar upload
│   │   ├── reset-password.tsx
│   │   ├── two-factor-challenge.tsx
│   │   └── verify-email.tsx
│   ├── forum/
│   │   ├── index.tsx
│   │   ├── show.tsx
│   │   └── thread/
│   │       └── show.tsx
│   ├── settings/
│   │   ├── appearance.tsx
│   │   ├── password.tsx
│   │   ├── profile.tsx                   # Con avatar y campos adicionales
│   │   └── two-factor.tsx
│   ├── dashboard.tsx
│   └── welcome.tsx                       # Landing page
├── routes/                               # Generado por Wayfinder
└── types/
    └── index.d.ts                        # Tipos globales
```

---

## 🔐 Sistema de Autenticación

### Flujo de Registro

1. Usuario llena formulario en `register.tsx`:
    - Nombre, email, password (obligatorios)
    - Avatar, edad, género, país, actividad enteógena (opcionales)
2. `CreateNewUser.php` procesa el registro:
    - Valida usando `ProfileValidationRules`
    - Guarda avatar en `storage/app/public/avatars/`
    - Asigna rol `user` automáticamente
3. `RegisterResponse.php` redirige según rol:
    - Admin → `/admin/dashboard`
    - Otros → `/`

### Login Social (Google)

```php
// routes/web.php
Route::get('auth/{provider}/redirect', [SocialAuthController::class, 'redirect']);
Route::get('auth/{provider}/callback', [SocialAuthController::class, 'callback']);
```

El `SocialAuthController`:

1. Redirige a Google para autenticación
2. En callback, busca o crea usuario
3. Guarda avatar de Google si no tiene local
4. Vincula cuenta social en `social_accounts`

### Variables de Entorno Necesarias

```env
GOOGLE_CLIENT_ID=tu-client-id
GOOGLE_CLIENT_SECRET=tu-client-secret
GOOGLE_REDIRECT_URI="${APP_URL}/auth/google/callback"
```

---

## 👥 Sistema de Roles y Permisos

### Roles

| Rol           | Slug      | Descripción                                       |
| ------------- | --------- | ------------------------------------------------- |
| Usuario       | `user`    | Colaborador básico. Auto-asignado al registrarse. |
| Curador       | `curator` | Representantes indígenas. Puede moderar.          |
| Administrador | `admin`   | Personal de Ser Ajayu. Gestión total.             |

### Permisos del Foro

| Permiso             | User | Curator | Admin |
| ------------------- | ---- | ------- | ----- |
| forum.view          | ✅   | ✅      | ✅    |
| forum.post          | ✅   | ✅      | ✅    |
| forum.comment       | ✅   | ✅      | ✅    |
| forum.vote          | ✅   | ✅      | ✅    |
| forum.create-thread | ✅   | ✅      | ✅    |
| forum.create-room   | ❌   | ✅      | ✅    |
| forum.moderate      | ❌   | ✅      | ✅    |
| forum.delete        | ❌   | ❌      | ✅    |

### Uso en Frontend

```tsx
// Verificar rol en componentes
const { auth } = usePage<SharedData>().props;
const isAdmin = auth.roles.includes('admin');
const isCurator = auth.roles.includes('curator');

// Renderizado condicional
{
    isAdmin && <Link href="/admin/dashboard">Dashboard Admin</Link>;
}
```

### Uso en Backend

```php
// En controllers
if (!$user->can('forum.moderate')) {
    return back()->withErrors(['error' => 'No tienes permisos']);
}

// En routes con middleware
Route::middleware(['role:admin'])->group(function () {
    Route::get('/admin/dashboard', [DashboardController::class, 'index']);
});
```

---

## 💬 Módulo de Foro

### Estructura de Datos

```
Forums (foros principales)
├── id, name, slug, description, icon
└── ForumThreads (hilos de discusión)
    ├── id, forum_id, user_id, title, is_sticky, is_locked
    └── ForumPosts (mensajes/respuestas)
        ├── id, thread_id, user_id, parent_id, content
        └── ForumVotes (votaciones)
            └── id, post_id, user_id, value (+1/-1)
```

### Rutas del Foro

```php
// routes/user.php (públicas para lectura)
Route::get('/foro', [ForumController::class, 'index']);
Route::get('/foro/{forum:slug}', [ForumController::class, 'show']);
Route::get('/foro/{forum:slug}/{thread:slug}', [ForumThreadController::class, 'show']);

// Requieren autenticación
Route::middleware('auth')->group(function () {
    Route::post('/foro/{forum:slug}/threads', [ForumThreadController::class, 'store']);
    Route::post('/foro/posts', [PostController::class, 'store']);
    Route::post('/foro/posts/{post}/vote', [PostController::class, 'vote']);
});

// Moderación (curator+)
Route::post('/foro/threads/{thread}/toggle-sticky', [ForumModerationController::class, 'toggleSticky']);
Route::post('/foro/threads/{thread}/toggle-lock', [ForumModerationController::class, 'toggleLock']);
Route::delete('/foro/threads/{thread}', [ForumModerationController::class, 'destroyThread']);
Route::delete('/foro/posts/{post}', [ForumModerationController::class, 'destroyPost']);
```

### Características

- **Acceso público**: Usuarios no autenticados pueden leer foros/threads
- **Login para participar**: Se requiere login para crear threads, responder, votar
- **Moderación**: Curators y admins pueden fijar, bloquear y eliminar
- **Notificaciones**: Al responder o votar, se notifica al autor del post

---

## 🔔 Sistema de Notificaciones

### Notificaciones Implementadas

1. **PostReply**: Cuando alguien responde a tu post
2. **PostUpvoted**: Cuando alguien da upvote a tu post

### Estructura de Payload

```php
// PostReply
[
    'type' => 'post_reply',
    'post_id' => $this->reply->id,
    'thread_id' => $this->reply->thread_id,
    'thread_slug' => $this->reply->thread->slug,
    'forum_slug' => $this->reply->thread->forum->slug,
    'replier_name' => $this->reply->user->name,
    'content_preview' => Str::limit($this->reply->content, 100),
]
```

### Frontend

El `NotificationsDropdown` muestra las notificaciones:

- Icono de campana con contador
- Lista de notificaciones no leídas
- Click para marcar como leída y navegar

---

## 🛤 Rutas de la Aplicación

### Estructura de Archivos de Rutas

| Archivo               | Middleware       | Descripción                |
| --------------------- | ---------------- | -------------------------- |
| `routes/web.php`      | -                | Públicas, landing, OAuth   |
| `routes/user.php`     | `auth` (parcial) | Foro, funciones de usuario |
| `routes/curator.php`  | `role:curator`   | Moderación                 |
| `routes/admin.php`    | `role:admin`     | Panel de administración    |
| `routes/settings.php` | `auth`           | Configuración de perfil    |

### Rutas Principales

```
GET  /                         → Landing page
GET  /foro                     → Lista de foros
GET  /foro/{slug}              → Ver foro con threads
GET  /foro/{forum}/{thread}    → Ver thread con posts
GET  /documentos               → Repositorio (placeholder)
GET  /encuestas                → Encuestas (placeholder)
GET  /proyectos                → Proyectos (placeholder)
GET  /mensajes                 → Mensajes (requiere auth, placeholder)

GET  /settings/profile         → Editar perfil
POST /settings/profile         → Actualizar perfil
POST /settings/profile/avatar  → Subir avatar
DELETE /settings/profile/avatar → Eliminar avatar

GET  /admin/dashboard          → Panel admin
GET  /admin/users              → Lista usuarios
```

---

## 🗃 Base de Datos

### Tabla `users`

```sql
id, name, email, email_verified_at, password,
avatar,                    -- Ruta local o URL externa (OAuth)
age,                       -- Edad (integer, nullable)
gender,                    -- Género (string, nullable)
country,                   -- País (string, nullable)
entheogen_activity,        -- Actividad enteógena (text, nullable)
two_factor_secret, two_factor_recovery_codes, two_factor_confirmed_at,
remember_token, created_at, updated_at
```

### Tabla `forums`

```sql
id, name, slug, description, icon, color, order, is_active, created_at, updated_at
```

### Tabla `forum_threads`

```sql
id, forum_id, user_id, title, slug, is_sticky, is_locked, views_count, created_at, updated_at
```

### Tabla `forum_posts`

```sql
id, thread_id, user_id, parent_id, content, is_solution, created_at, updated_at
```

### Tabla `forum_votes`

```sql
id, post_id, user_id, value (+1/-1), created_at, updated_at
-- Unique constraint on (post_id, user_id)
```

### Tabla `social_accounts`

```sql
id, user_id, provider, provider_id, provider_token, provider_refresh_token, avatar_url, created_at, updated_at
```

---

## 🎨 Componentes UI Clave

### UserDropdown

```tsx
// Muestra avatar del usuario con menú desplegable
// Incluye links condicionados por rol:
// - Dashboard Admin (si admin)
// - Panel de Curador (si curator)
// - Editar Perfil
// - Cerrar Sesión
```

### PasswordInput

```tsx
// Input de contraseña con toggle de visibilidad
// Usado en login, register, reset-password, confirm-password
```

### VoteControl

```tsx
// Componente de votación para posts del foro
// Muestra flechas up/down con contador
// Redirige a login si no autenticado
```

### NotificationsDropdown

```tsx
// Dropdown con campana y contador de notificaciones
// Lista notificaciones no leídas
// Mark as read individual y masivo
```

---

## ⏳ Pendientes y Próximos Pasos

### Alta Prioridad

1. **Repositorio de Documentos** (Parcialmente Implementado)
    - ✅ Vista dual: Galería Curada y Carpeta Google Drive
    - ✅ Integración con Google Drive vía iframe (público)
    - ✅ Visor de documentos en modal
    - ⏳ Sincronización automática vía API (Backend)

2. **Panel de Administración Completo**
    - CRUD de usuarios
    - Gestión de roles
    - Estadísticas

3. **Encuestas Digitales**
    - Crear encuestas
    - Responder encuestas
    - Visualización de resultados

### Media Prioridad

4. **Documentos Colaborativos**
    - Editor en tiempo real (considerar TipTap o similar)
    - Moderación de ediciones

5. **Mensajería**
    - Chat privado
    - Grupos temáticos
    - Cifrado end-to-end

### Baja Prioridad

6. **Multilenguaje**
    - Español (actual)
    - Inglés
    - Quechua/Aymara

7. **PWA/Mobile**
    - Service Worker
    - Push notifications

8. **Blockchain/NFTs**
    - Certificación de participación
    - Trazabilidad

---

## 🐛 Problemas Conocidos y Soluciones

### 1. Imágenes de Avatar No Cargan

**Problema**: Las fotos de perfil no se muestran después de subir.

**Solución**:

```bash
php artisan storage:link
```

Además, en `User.php` se usa URL relativa para evitar problemas con `APP_URL`:

```php
return '/storage/' . $this->avatar;
```

### 2. PostTooLargeException al Subir Avatar

**Problema**: Error 413 al subir imágenes grandes.

**Solución**: Validación en frontend antes de enviar:

```tsx
if (file.size > 2 * 1024 * 1024) {
    setError('avatar', 'La imagen no debe pesar más de 2MB.');
    return;
}
```

### 3. TypeScript Errors con Wayfinder Form

**Problema**: Tipos estrictos en render props del Form component.

**Solución temporal**: Usar `any` en desestructuración:

```tsx
{({ processing, errors, setData }: any) => { ... }}
```

### 4. Roles No Se Muestran en Frontend

**Problema**: `user.roles` undefined.

**Solución**: Los roles se pasan en `auth.roles` como array de strings, no en el objeto user:

```tsx
const { auth } = usePage<SharedData>().props;
const isAdmin = auth.roles.includes('admin');
```

---

## 📚 Recursos y Referencias

### Documentación

- [Laravel 12 Docs](https://laravel.com/docs/12.x)
- [Inertia.js v2 Docs](https://inertiajs.com/)
- [Spatie Permission Docs](https://spatie.be/docs/laravel-permission)
- [Pest Testing Docs](https://pestphp.com/)

### Archivos del Proyecto

- `.agent/rules.md` - Reglas y convenciones
- `.agent/workflows/` - Workflows automatizados
- `docs/information.md` - Ficha técnica
- `docs/Pantallas.png` - Wireframes

### Comandos Útiles

```bash
# Desarrollo
composer run dev          # Inicia servidor + vite
npm run dev               # Solo Vite

# Base de datos
php artisan migrate:fresh --seed  # Reset DB
php artisan db:seed --class=ForumSeeder  # Seed específico

# Testing
php artisan test --compact
php artisan test --filter=ForumTest

# Utilidades
php artisan wayfinder:generate   # Regenerar rutas TS
vendor/bin/pint                  # Formatear código
php artisan route:list           # Ver rutas
```

---

## 🔄 Historial de Cambios Recientes

| Fecha      | Cambio                                     |
| ---------- | ------------------------------------------ |
| 2026-01-20 | Documentación completa actualizada         |
| 2026-01-17 | Fix avatar URL relativa, storage:link      |
| 2026-01-17 | Validación tamaño archivo frontend         |
| 2026-01-17 | Profile page con PublicLayout              |
| 2026-01-17 | Sistema de notificaciones foro             |
| 2026-01-17 | Login social Google implementado           |
| 2026-01-17 | Foro completo con votaciones y moderación  |
| 2026-01-16 | Sistema de roles y permisos                |
| 2026-01-16 | Formulario registro con campos adicionales |
| 2026-01-16 | Landing page con animaciones               |
