import { useForm } from '@inertiajs/react';
import { useState } from 'react';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { toast } from 'sonner';
import { store } from '@/actions/App/Http/Controllers/Forum/ThreadController';

interface Props {
    open: boolean;
    onOpenChange: (open: boolean) => void;
    forumSlug: string;
}

export function CreateThreadModal({ open, onOpenChange, forumSlug }: Props) {
    const { data, setData, post, processing, errors, reset } = useForm({
        title: '',
        content: '',
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        post(store.url(forumSlug), {
            onSuccess: () => {
                onOpenChange(false);
                reset();
                toast.success('Hilo creado exitosamente');
            },
            onError: () => {
                toast.error('Error al crear el hilo. Por favor revisa los campos.');
            },
        });
    };

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent className="sm:max-w-[600px]">
                <DialogHeader>
                    <DialogTitle>Crear Nuevo Hilo de Discusión</DialogTitle>
                    <DialogDescription>
                        Inicia un nuevo tema de conversación en este foro. Sé respetuoso y claro.
                    </DialogDescription>
                </DialogHeader>
                <form onSubmit={handleSubmit} className="space-y-4">
                    <div className="space-y-2">
                        <Label htmlFor="title">Título del Hilo</Label>
                        <Input
                            id="title"
                            value={data.title}
                            onChange={(e) => setData('title', e.target.value)}
                            placeholder="Ej: Propuesta para regulación de..."
                            required
                        />
                        {errors.title && <p className="text-sm text-red-500">{errors.title}</p>}
                    </div>
                    <div className="space-y-2">
                        <Label htmlFor="content">Contenido</Label>
                        <Textarea
                            id="content"
                            value={data.content}
                            onChange={(e) => setData('content', e.target.value)}
                            placeholder="Describe tu propuesta o tema de discusión..."
                            className="min-h-[200px]"
                            required
                        />
                        {errors.content && <p className="text-sm text-red-500">{errors.content}</p>}
                    </div>
                    <DialogFooter>
                        <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
                            Cancelar
                        </Button>
                        <Button type="submit" disabled={processing} className="bg-forest hover:bg-forest-light">
                            {processing ? 'Publicando...' : 'Publicar Hilo'}
                        </Button>
                    </DialogFooter>
                </form>
            </DialogContent>
        </Dialog>
    );
}
