import { Link, usePage } from '@inertiajs/react';
import { motion, AnimatePresence } from 'framer-motion';
import { 
    Home, 
    FileText, 
    MessageSquare, 
    BarChart3, 
    Mail,
    type LucideIcon
} from 'lucide-react';
import { cn } from '@/lib/utils';

interface NavItem {
    name: string;
    href: string;
    icon: LucideIcon;
}

const navItems: NavItem[] = [
    { name: 'Inicio', href: '/', icon: Home },
    { name: 'Documentos', href: '/documentos', icon: FileText },
    { name: 'Foro', href: '/foro', icon: MessageSquare },
    { name: 'Encuestas', href: '/encuestas', icon: BarChart3 },
    { name: 'Mensajes', href: '/mensajes', icon: Mail },
];

export function BottomNav() {
    const { url } = usePage();

    const isActive = (href: string) => {
        if (href === '/') {
            return url === '/';
        }
        return url.startsWith(href);
    };

    return (
        <motion.nav 
            className="fixed bottom-0 left-0 right-0 z-50 md:hidden"
            initial={{ y: 100 }}
            animate={{ y: 0 }}
            transition={{ type: 'spring', stiffness: 300, damping: 30 }}
        >
            {/* Shadow overlay */}
            <div className="absolute -top-6 left-0 right-0 h-6 bg-gradient-to-t from-background/80 to-transparent pointer-events-none" />
            
            {/* Main container with solid background */}
            <div className="bg-card border-t border-border shadow-[0_-4px_20px_rgba(0,0,0,0.08)]">
                <div className="flex items-center justify-around h-16 px-2 safe-area-pb">
                    {navItems.map((item, index) => {
                        const Icon = item.icon;
                        const active = isActive(item.href);

                        return (
                            <Link
                                key={item.name}
                                href={item.href}
                                className="relative flex-1 flex items-center justify-center"
                            >
                                <motion.div
                                    className="relative flex flex-col items-center justify-center py-2 px-3"
                                    initial={{ opacity: 0, y: 10 }}
                                    animate={{ opacity: 1, y: 0 }}
                                    transition={{ delay: index * 0.05 }}
                                    whileTap={{ scale: 0.9 }}
                                >
                                    {/* Active background pill */}
                                    <AnimatePresence>
                                        {active && (
                                            <motion.div
                                                className="absolute inset-0 bg-forest/10 rounded-2xl"
                                                layoutId="bottomNavActiveBackground"
                                                initial={{ opacity: 0, scale: 0.8 }}
                                                animate={{ opacity: 1, scale: 1 }}
                                                exit={{ opacity: 0, scale: 0.8 }}
                                                transition={{ 
                                                    type: 'spring', 
                                                    stiffness: 500, 
                                                    damping: 35 
                                                }}
                                            />
                                        )}
                                    </AnimatePresence>

                                    {/* Icon container */}
                                    <motion.div
                                        className="relative z-10"
                                        animate={{ 
                                            scale: active ? 1.1 : 1,
                                            y: active ? -2 : 0
                                        }}
                                        transition={{ type: 'spring', stiffness: 400 }}
                                    >
                                        <Icon 
                                            className={cn(
                                                'h-6 w-6 transition-colors duration-200',
                                                active 
                                                    ? 'text-forest' 
                                                    : 'text-muted-foreground'
                                            )} 
                                            strokeWidth={active ? 2.5 : 2}
                                        />
                                    </motion.div>

                                    {/* Label */}
                                    <motion.span 
                                        className={cn(
                                            'relative z-10 text-[11px] font-medium mt-1 transition-colors duration-200',
                                            active 
                                                ? 'text-forest' 
                                                : 'text-muted-foreground'
                                        )}
                                        animate={{ 
                                            fontWeight: active ? 600 : 500
                                        }}
                                    >
                                        {item.name}
                                    </motion.span>

                                    {/* Active dot indicator */}
                                    <AnimatePresence>
                                        {active && (
                                            <motion.div
                                                className="absolute -top-1 left-1/2 -translate-x-1/2 w-1.5 h-1.5 bg-forest rounded-full"
                                                initial={{ opacity: 0, scale: 0 }}
                                                animate={{ opacity: 1, scale: 1 }}
                                                exit={{ opacity: 0, scale: 0 }}
                                                transition={{ type: 'spring', stiffness: 500 }}
                                            />
                                        )}
                                    </AnimatePresence>
                                </motion.div>
                            </Link>
                        );
                    })}
                </div>
            </div>
        </motion.nav>
    );
}

export default BottomNav;
