import { Link, router, usePage } from '@inertiajs/react';
import { motion } from 'framer-motion';
import { 
    LogOut, 
    User, 
    Settings, 
    LayoutDashboard,
    ChevronDown,
} from 'lucide-react';
import { Button } from '@/components/ui/button';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuLabel,
    DropdownMenuSeparator,
    DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { type SharedData } from '@/types';

export function UserDropdown() {
    const { auth } = usePage<SharedData>().props;
    const user = auth.user;
    const roles = auth.roles;

    if (!user) return null;

    const isAdmin = roles.includes('admin');
    const isCurator = roles.includes('curator');

    const handleLogout = () => {
        router.post('/logout');
    };

    // Get user initials for avatar fallback
    const getInitials = (name: string) => {
        return name
            .split(' ')
            .map(word => word[0])
            .join('')
            .toUpperCase()
            .slice(0, 2);
    };

    return (
        <DropdownMenu>
            <DropdownMenuTrigger asChild>
                <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                    <Button 
                        variant="outline" 
                        size="sm" 
                        className="gap-2 border-forest/30 hover:bg-forest/10 hover:border-forest"
                    >
                        <Avatar className="h-6 w-6">
                            <AvatarImage src={user.profile_photo_url ?? undefined} alt={user.name} />
                            <AvatarFallback className="text-xs bg-forest/20 text-forest">
                                {getInitials(user.name)}
                            </AvatarFallback>
                        </Avatar>
                        <span className="hidden sm:inline max-w-[100px] truncate">{user.name}</span>
                        <ChevronDown className="h-3 w-3 text-muted-foreground" />
                    </Button>
                </motion.div>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end" className="w-56">
                <DropdownMenuLabel className="font-normal">
                    <div className="flex flex-col space-y-1">
                        <p className="text-sm font-medium leading-none">{user.name}</p>
                        <p className="text-xs leading-none text-muted-foreground">
                            {user.email}
                        </p>
                    </div>
                </DropdownMenuLabel>
                <DropdownMenuSeparator />

                {/* Admin Dashboard Link */}
                {isAdmin && (
                    <DropdownMenuItem asChild>
                        <Link href="/admin/dashboard" className="w-full cursor-pointer">
                            <LayoutDashboard className="mr-2 h-4 w-4" />
                            Dashboard Admin
                        </Link>
                    </DropdownMenuItem>
                )}

                {/* Curator Dashboard Link */}
                {isCurator && !isAdmin && (
                    <DropdownMenuItem asChild>
                        <Link href="/curator" className="w-full cursor-pointer">
                            <LayoutDashboard className="mr-2 h-4 w-4" />
                            Panel de Curador
                        </Link>
                    </DropdownMenuItem>
                )}

                {/* Profile Settings */}
                <DropdownMenuItem asChild>
                    <Link href="/settings/profile" className="w-full cursor-pointer">
                        <User className="mr-2 h-4 w-4" />
                        Editar perfil
                    </Link>
                </DropdownMenuItem>

                {/* Settings */}
                <DropdownMenuItem asChild>
                    <Link href="/settings" className="w-full cursor-pointer">
                        <Settings className="mr-2 h-4 w-4" />
                        Configuración
                    </Link>
                </DropdownMenuItem>

                <DropdownMenuSeparator />

                {/* Logout */}
                <DropdownMenuItem 
                    onClick={handleLogout}
                    className="text-destructive focus:text-destructive cursor-pointer"
                >
                    <LogOut className="mr-2 h-4 w-4" />
                    Cerrar sesión
                </DropdownMenuItem>
            </DropdownMenuContent>
        </DropdownMenu>
    );
}
