import { type PropsWithChildren } from 'react';
import { Link, usePage } from '@inertiajs/react';
import { motion, AnimatePresence } from 'framer-motion';
import { Globe, Menu } from 'lucide-react';
import AppearanceToggleDropdown from '@/components/appearance-dropdown';
import { NotificationsDropdown } from '@/components/notifications-dropdown';
import { UserDropdown } from '@/components/user-dropdown';
import { BottomNav } from '@/components/navigation/bottom-nav';
import { Button } from '@/components/ui/button';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
    Sheet,
    SheetContent,
    SheetHeader,
    SheetTitle,
    SheetTrigger,
} from '@/components/ui/sheet';
import { type SharedData } from '@/types';
import { cn } from '@/lib/utils';

interface PublicLayoutProps extends PropsWithChildren {
    showBottomNav?: boolean;
    className?: string;
}

// Navigation links according to the technical document
const navLinks = [
    { name: 'Inicio', href: '/' },
    { name: 'Documentos', href: '/documentos' },
    { name: 'Foro', href: '/foro' },
    { name: 'Encuestas', href: '/encuestas' },
    { name: 'Proyectos', href: '/proyectos' },
];

function Logo() {
    return (
        <Link href="/" className="flex items-center gap-2.5 group">
            <motion.div 
                className="flex h-10 w-10 items-center justify-center rounded-xl bg-gradient-to-br from-forest to-forest-light text-white shadow-lg shadow-forest/20"
                whileHover={{ scale: 1.05, rotate: 5 }}
                whileTap={{ scale: 0.95 }}
                transition={{ type: 'spring', stiffness: 400, damping: 17 }}
            >
                <svg
                    viewBox="0 0 24 24"
                    fill="none"
                    className="h-6 w-6"
                    stroke="currentColor"
                    strokeWidth="1.8"
                >
                    <path d="M12 2L21.5 7.5V16.5L12 22L2.5 16.5V7.5L12 2Z" />
                    <circle cx="12" cy="12" r="3.5" />
                </svg>
            </motion.div>
            <span className="text-xl font-bold bg-gradient-to-r from-forest to-forest-light bg-clip-text text-transparent group-hover:from-forest-light group-hover:to-forest transition-all duration-300">
                Ser Ajayu
            </span>
        </Link>
    );
}

function LanguageSelector() {
    return (
        <DropdownMenu>
            <DropdownMenuTrigger asChild>
                <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                    <Button variant="ghost" size="sm" className="gap-2 hover:bg-forest/10">
                        <Globe className="h-4 w-4" />
                        <span className="hidden sm:inline font-medium">ES</span>
                    </Button>
                </motion.div>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end" className="min-w-[160px]">
                <DropdownMenuItem className="cursor-pointer">
                    <span className="mr-2">🇪🇸</span> Español
                </DropdownMenuItem>
                <DropdownMenuItem className="cursor-pointer">
                    <span className="mr-2">🇬🇧</span> English
                </DropdownMenuItem>
                <DropdownMenuItem disabled className="opacity-50">
                    <span className="mr-2">🏔️</span> Quechua (próx.)
                </DropdownMenuItem>
            </DropdownMenuContent>
        </DropdownMenu>
    );
}

function DesktopNav() {
    const { url } = usePage();

    return (
        <nav className="hidden lg:flex items-center gap-1">
            {navLinks.map((link, index) => {
                const isActive = link.href === '/' 
                    ? url === '/' 
                    : url.startsWith(link.href);

                return (
                    <motion.div
                        key={link.name}
                        initial={{ opacity: 0, y: -10 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ delay: index * 0.05 }}
                    >
                        <Link
                            href={link.href}
                            className={cn(
                                'relative px-4 py-2 text-sm font-medium rounded-lg transition-all duration-300',
                                isActive
                                    ? 'text-forest'
                                    : 'text-muted-foreground hover:text-foreground hover:bg-muted/50'
                            )}
                        >
                            {link.name}
                            {isActive && (
                                <motion.div
                                    className="absolute bottom-0 left-1/2 -translate-x-1/2 w-6 h-0.5 bg-forest rounded-full"
                                    layoutId="desktopActiveIndicator"
                                    transition={{ type: 'spring', stiffness: 500, damping: 30 }}
                                />
                            )}
                        </Link>
                    </motion.div>
                );
            })}
        </nav>
    );
}

function MobileMenu() {
    const { url } = usePage();

    return (
        <Sheet>
            <SheetTrigger asChild>
                <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                    <Button variant="ghost" size="icon" className="lg:hidden">
                        <Menu className="h-5 w-5" />
                    </Button>
                </motion.div>
            </SheetTrigger>
            <SheetContent side="left" className="w-80 bg-background/95 backdrop-blur-xl">
                <SheetHeader className="pb-6 border-b border-border">
                    <SheetTitle>
                        <Logo />
                    </SheetTitle>
                </SheetHeader>
                <nav className="mt-6 flex flex-col gap-1">
                    {navLinks.map((link, index) => {
                        const isActive = link.href === '/' 
                            ? url === '/' 
                            : url.startsWith(link.href);

                        return (
                            <motion.div
                                key={link.name}
                                initial={{ opacity: 0, x: -20 }}
                                animate={{ opacity: 1, x: 0 }}
                                transition={{ delay: index * 0.05 }}
                            >
                                <Link
                                    href={link.href}
                                    className={cn(
                                        'flex items-center px-4 py-3 text-base font-medium rounded-xl transition-all duration-300',
                                        isActive
                                            ? 'text-forest bg-forest/10'
                                            : 'text-muted-foreground hover:text-foreground hover:bg-muted/50'
                                    )}
                                >
                                    {link.name}
                                </Link>
                            </motion.div>
                        );
                    })}
                </nav>
            </SheetContent>
        </Sheet>
    );
}


function AuthButtons() {
    const { auth } = usePage<SharedData>().props;

    if (auth.user) {
        return <UserDropdown />;
    }

    return (
        <div className="flex items-center gap-2">
            <Link href="/login">
                <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                    <Button variant="ghost" size="sm" className="hover:bg-forest/10 hover:text-forest">
                        Ingresar
                    </Button>
                </motion.div>
            </Link>
            <Link href="/register" className="hidden sm:block">
                <motion.div whileHover={{ scale: 1.02 }} whileTap={{ scale: 0.98 }}>
                    <Button size="sm" className="bg-forest hover:bg-forest-light shadow-lg shadow-forest/20">
                        Registrarse
                    </Button>
                </motion.div>
            </Link>
        </div>
    );
}

function Header() {
    return (
        <motion.header 
            className="sticky top-0 z-40 w-full border-b border-border/50 bg-background/80 backdrop-blur-xl"
            initial={{ y: -100 }}
            animate={{ y: 0 }}
            transition={{ type: 'spring', stiffness: 300, damping: 30 }}
        >
            <div className="container mx-auto flex h-16 items-center justify-between px-4">
                {/* Left: Mobile menu + Logo */}
                <div className="flex items-center gap-2">
                    <MobileMenu />
                    <Logo />
                </div>

                {/* Center: Desktop Navigation */}
                <DesktopNav />

                {/* Right: Theme + Language + Auth */}
                <div className="flex items-center gap-1">
                    <AppearanceToggleDropdown />
                    <LanguageSelector />
                    <NotificationsDropdown />
                    <AuthButtons />
                </div>
            </div>
        </motion.header>
    );
}

function Footer() {
    return (
        <footer className="border-t border-border bg-muted/30 pb-24 md:pb-8">
            <div className="container mx-auto px-4 py-12">
                <div className="grid gap-8 md:grid-cols-4">
                    {/* Logo & Description */}
                    <div className="md:col-span-2">
                        <Logo />
                        <p className="mt-4 text-sm text-muted-foreground max-w-md leading-relaxed">
                            Integrando saberes ancestrales andinos y amazónicos con
                            conocimiento científico moderno para la salud mental, de manera
                            colaborativa, respetuosa y regulada.
                        </p>
                    </div>

                    {/* Links */}
                    <div>
                        <h4 className="font-semibold text-foreground mb-4">Explorar</h4>
                        <ul className="space-y-3 text-sm text-muted-foreground">
                            <li><Link href="/documentos" className="hover:text-forest transition-colors">Documentos</Link></li>
                            <li><Link href="/foro" className="hover:text-forest transition-colors">Foro</Link></li>
                            <li><Link href="/encuestas" className="hover:text-forest transition-colors">Encuestas</Link></li>
                            <li><Link href="/proyectos" className="hover:text-forest transition-colors">Proyectos</Link></li>
                        </ul>
                    </div>

                    {/* Legal */}
                    <div>
                        <h4 className="font-semibold text-foreground mb-4">Legal</h4>
                        <ul className="space-y-3 text-sm text-muted-foreground">
                            <li><Link href="/privacidad" className="hover:text-forest transition-colors">Política de Privacidad</Link></li>
                            <li><Link href="/terminos" className="hover:text-forest transition-colors">Términos de Servicio</Link></li>
                            <li><Link href="/reglamento" className="hover:text-forest transition-colors">Reglamento</Link></li>
                        </ul>
                    </div>
                </div>

                <div className="mt-12 pt-8 border-t border-border flex flex-col md:flex-row justify-between items-center gap-4">
                    <p className="text-sm text-muted-foreground">
                        © {new Date().getFullYear()} Ser Ajayu. Todos los derechos reservados.
                    </p>
                    <div className="flex items-center gap-6">
                        <a 
                            href="https://serajayu.com" 
                            target="_blank" 
                            rel="noopener noreferrer"
                            className="text-sm text-muted-foreground hover:text-forest transition-colors"
                        >
                            serajayu.com
                        </a>
                    </div>
                </div>
            </div>
        </footer>
    );
}

export default function PublicLayout({ 
    children, 
    showBottomNav = true,
    className 
}: PublicLayoutProps) {
    return (
        <div className="min-h-screen flex flex-col bg-background">
            <Header />
            
            <main className={cn("flex-1", showBottomNav && "pb-16 md:pb-0", className)}>
                {children}
            </main>

            <Footer />

            <AnimatePresence>
                {showBottomNav && <BottomNav />}
            </AnimatePresence>
        </div>
    );
}
