import { Head, Link } from '@inertiajs/react';
import { motion } from 'framer-motion';
import { 
    Users, 
    FileText, 
    BarChart3, 
    Settings,
    Shield,
    ChevronRight,
    UserPlus,
    Activity
} from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';

interface DashboardStats {
    totalUsers: number;
    totalDocuments: number;
    totalSurveys: number;
    pendingModerations: number;
}

interface Props {
    stats?: DashboardStats;
}

const defaultStats: DashboardStats = {
    totalUsers: 0,
    totalDocuments: 7,
    totalSurveys: 0,
    pendingModerations: 0,
};

const adminMenuItems = [
    {
        title: 'Gestión de Usuarios',
        description: 'Ver, crear y asignar roles a usuarios',
        icon: Users,
        href: '/admin/users',
        color: 'bg-forest/10 text-forest',
    },
    {
        title: 'Documentos',
        description: 'Administrar repositorio de documentos',
        icon: FileText,
        href: '/admin/documents',
        color: 'bg-ochre/10 text-ochre',
    },
    {
        title: 'Encuestas',
        description: 'Crear y gestionar encuestas',
        icon: BarChart3,
        href: '/admin/surveys',
        color: 'bg-emerald/10 text-emerald',
    },
    {
        title: 'Configuración',
        description: 'Ajustes de la aplicación',
        icon: Settings,
        href: '/admin/settings',
        color: 'bg-terracotta/10 text-terracotta',
    },
];

export default function AdminDashboard({ stats = defaultStats }: Props) {
    return (
        <AppLayout>
            <Head title="Panel de Administración" />

            <div className="p-6 space-y-8">
                {/* Header */}
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    className="flex items-center justify-between"
                >
                    <div className="flex items-center gap-4">
                        <div className="p-3 rounded-2xl bg-forest/10">
                            <Shield className="h-8 w-8 text-forest" />
                        </div>
                        <div>
                            <h1 className="text-2xl font-bold text-foreground">
                                Panel de Administración
                            </h1>
                            <p className="text-muted-foreground">
                                Gestión completa de Ser Ajayu
                            </p>
                        </div>
                    </div>

                    <Link href="/admin/users/create">
                        <Button className="bg-forest hover:bg-forest-light gap-2">
                            <UserPlus className="h-4 w-4" />
                            Nuevo Usuario
                        </Button>
                    </Link>
                </motion.div>

                {/* Stats Cards */}
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.1 }}
                    className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4"
                >
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">
                                Usuarios Registrados
                            </CardTitle>
                            <Users className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold">{stats.totalUsers}</div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">
                                Documentos
                            </CardTitle>
                            <FileText className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold">{stats.totalDocuments}</div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">
                                Encuestas Activas
                            </CardTitle>
                            <BarChart3 className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold">{stats.totalSurveys}</div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">
                                Pendientes de Moderación
                            </CardTitle>
                            <Activity className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold">{stats.pendingModerations}</div>
                        </CardContent>
                    </Card>
                </motion.div>

                {/* Quick Actions */}
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.2 }}
                >
                    <h2 className="text-lg font-semibold mb-4">Acciones Rápidas</h2>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        {adminMenuItems.map((item, index) => (
                            <motion.div
                                key={item.title}
                                initial={{ opacity: 0, y: 20 }}
                                animate={{ opacity: 1, y: 0 }}
                                transition={{ delay: 0.1 * (index + 3) }}
                            >
                                <Link href={item.href}>
                                    <Card className="hover:shadow-md transition-shadow cursor-pointer group">
                                        <CardHeader className="flex flex-row items-center gap-4">
                                            <div className={`p-3 rounded-xl ${item.color}`}>
                                                <item.icon className="h-6 w-6" />
                                            </div>
                                            <div className="flex-1">
                                                <CardTitle className="text-base group-hover:text-forest transition-colors">
                                                    {item.title}
                                                </CardTitle>
                                                <CardDescription>
                                                    {item.description}
                                                </CardDescription>
                                            </div>
                                            <ChevronRight className="h-5 w-5 text-muted-foreground group-hover:text-forest group-hover:translate-x-1 transition-all" />
                                        </CardHeader>
                                    </Card>
                                </Link>
                            </motion.div>
                        ))}
                    </div>
                </motion.div>
            </div>
        </AppLayout>
    );
}
