import { Head, Link } from '@inertiajs/react';
import { motion } from 'framer-motion';
import { show } from '@/actions/App/Http/Controllers/Forum/ForumController';
import { MessageSquare, Users, Clock, ArrowRight, MessageCircle } from 'lucide-react';
import PublicLayout from '@/layouts/public-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Separator } from '@/components/ui/separator';

interface Forum {
    id: number;
    name: string;
    slug: string;
    description: string;
    threads_count: number;
    posts_count: number;
    latest_thread?: {
        title: string;
        created_at: string;
        user: {
            name: string;
            profile_photo_url?: string;
        };
    };
}

interface Props {
    forums: Forum[];
}

export default function ForumIndex({ forums }: Props) {
    return (
        <PublicLayout>
            <Head title="Foros de Comunidad" />

            <div className="max-w-5xl mx-auto p-6 space-y-8">
                {/* Header */}
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    className="text-center space-y-4"
                >
                    <h1 className="text-4xl font-bold font-heading text-forest-dark">
                        Foros de Comunidad
                    </h1>
                    <p className="text-xl text-muted-foreground max-w-2xl mx-auto">
                        Espacios de diálogo para construir colectivamente saberes, políticas y proyectos.
                    </p>
                </motion.div>

                {/* Forum List */}
                <div className="grid gap-6">
                    {forums.map((forum, index) => (
                        <motion.div
                            key={forum.id}
                            initial={{ opacity: 0, y: 20 }}
                            animate={{ opacity: 1, y: 0 }}
                            transition={{ delay: index * 0.1 }}
                        >
                            <Link href={show.url(forum.slug)}>
                                <Card className="hover:shadow-lg transition-all duration-300 group border-l-4 border-l-transparent hover:border-l-forest">
                                    <div className="flex flex-col md:flex-row gap-6 p-6">
                                        {/* Icon Section */}
                                        <div className="hidden md:flex flex-col items-center justify-center w-24 shrink-0">
                                            <div className="p-4 rounded-full bg-forest/10 group-hover:bg-forest/20 transition-colors">
                                                <MessageSquare className="h-8 w-8 text-forest" />
                                            </div>
                                        </div>

                                        {/* Content Section */}
                                        <div className="flex-1 space-y-4">
                                            <div>
                                                <CardTitle className="text-xl text-forest-dark mb-2 group-hover:text-forest transition-colors flex items-center gap-2">
                                                    {forum.name}
                                                    <ArrowRight className="h-4 w-4 opacity-0 -translate-x-2 group-hover:opacity-100 group-hover:translate-x-0 transition-all" />
                                                </CardTitle>
                                                <CardDescription className="text-base">
                                                    {forum.description}
                                                </CardDescription>
                                            </div>

                                            {/* Stats & Metadata - Mobile Only */}
                                            <div className="md:hidden flex items-center gap-4 text-sm text-muted-foreground">
                                                <div className="flex items-center gap-1">
                                                    <MessageSquare className="h-4 w-4" />
                                                    {forum.threads_count}
                                                </div>
                                                <div className="flex items-center gap-1">
                                                    <MessageCircle className="h-4 w-4" />
                                                    {forum.posts_count}
                                                </div>
                                            </div>

                                            {/* Latest Activity Section */}
                                            {forum.latest_thread && (
                                                <div className="bg-muted/30 rounded-lg p-3 text-sm">
                                                    <div className="flex items-center gap-2 text-muted-foreground mb-1">
                                                        <Clock className="h-3 w-3" />
                                                        <span>Última actividad</span>
                                                    </div>
                                                    <div className="flex items-center justify-between gap-4">
                                                        <span className="font-medium text-foreground truncate">
                                                            {forum.latest_thread.title}
                                                        </span>
                                                        <div className="flex items-center gap-2 shrink-0">
                                                            <span className="text-xs text-muted-foreground">
                                                                por {forum.latest_thread.user.name}
                                                            </span>
                                                            <Avatar className="h-5 w-5">
                                                                <AvatarImage src={forum.latest_thread.user.profile_photo_url} />
                                                                <AvatarFallback>{forum.latest_thread.user.name.charAt(0)}</AvatarFallback>
                                                            </Avatar>
                                                        </div>
                                                    </div>
                                                </div>
                                            )}
                                        </div>

                                        {/* Desktop Stats */}
                                        <div className="hidden md:flex flex-col justify-center items-end gap-2 min-w-[120px] border-l pl-6">
                                            <div className="text-right">
                                                <div className="text-2xl font-bold text-forest">{forum.threads_count}</div>
                                                <div className="text-xs text-muted-foreground uppercase tracking-wider">Hilos</div>
                                            </div>
                                            <Separator />
                                            <div className="text-right">
                                                <div className="text-lg font-bold text-muted-foreground">{forum.posts_count}</div>
                                                <div className="text-xs text-muted-foreground uppercase tracking-wider">Comentarios</div>
                                            </div>
                                        </div>
                                    </div>
                                </Card>
                            </Link>
                        </motion.div>
                    ))}
                </div>
            </div>
        </PublicLayout>
    );
}
