import { Head, Link } from '@inertiajs/react';
import { motion } from 'framer-motion';
import { 
    Sparkles, 
    ChevronRight, 
    ExternalLink,
    Music,
    Video,
    Globe
} from 'lucide-react';
import PublicLayout from '@/layouts/public-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';

const fadeInUp = {
    hidden: { opacity: 0, y: 20 },
    visible: { 
        opacity: 1, 
        y: 0,
        transition: { duration: 0.4, ease: [0.22, 1, 0.36, 1] }
    }
};

const staggerContainer = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.1,
            delayChildren: 0.2
        }
    }
};

const projects = [
    {
        id: '1',
        title: 'Proyecto Sonoro',
        description: 'Plantas sagradas y su dimensión sonora en las tradiciones andinas y amazónicas. Documentación de cantos ceremoniales, ícaros y música tradicional asociada a prácticas terapéuticas.',
        icon: Music,
        status: 'active' as const,
        href: 'https://giveth.io/project/sound-of-sacred-plants-in-the-face-of-modern-crises',
        color: 'forest',
    },
    {
        id: '2',
        title: 'Hub de Conocimiento',
        description: 'Proyecto marco para la integración de saberes ancestrales con evidencia científica moderna. Plataforma colaborativa para la generación de conocimiento intercultural.',
        icon: Globe,
        status: 'development' as const,
        color: 'ochre',
    },
    {
        id: '3',
        title: 'Proyecto Video',
        description: 'Documentación audiovisual de prácticas tradicionales y testimonios de comunidades indígenas. Archivo de memoria visual para la preservación de conocimientos.',
        icon: Video,
        status: 'soon' as const,
        color: 'terracotta',
    },
];

const statusConfig = {
    active: { label: 'Activo', class: 'bg-forest/20 text-forest border-forest/30' },
    development: { label: 'En desarrollo', class: 'bg-ochre/20 text-ochre border-ochre/30' },
    soon: { label: 'Próximamente', class: 'bg-terracotta/20 text-terracotta border-terracotta/30' },
};

const colorClasses = {
    forest: 'bg-forest/10 text-forest',
    ochre: 'bg-ochre/10 text-ochre',
    terracotta: 'bg-terracotta/10 text-terracotta',
};

export default function ProyectosIndex() {
    return (
        <PublicLayout>
            <Head title="Proyectos - Ser Ajayu" />

            <section className="relative py-16 lg:py-24 overflow-hidden">
                <div className="absolute inset-0 -z-10">
                    <div className="absolute top-0 left-1/4 h-64 w-64 rounded-full bg-forest/10 blur-[100px]" />
                    <div className="absolute bottom-0 right-1/4 h-48 w-48 rounded-full bg-ochre/10 blur-[80px]" />
                </div>

                <div className="container mx-auto px-4">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.6 }}
                        className="max-w-3xl"
                    >
                        <nav className="flex items-center gap-2 text-sm text-muted-foreground mb-6">
                            <Link href="/" className="hover:text-foreground transition-colors">Inicio</Link>
                            <ChevronRight className="h-4 w-4" />
                            <span className="text-foreground">Proyectos</span>
                        </nav>

                        <div className="flex items-center gap-4 mb-4">
                            <div className="p-3 rounded-2xl bg-forest/10">
                                <Sparkles className="h-8 w-8 text-forest" />
                            </div>
                            <div>
                                <h1 className="text-3xl lg:text-4xl font-bold text-foreground">
                                    Nuestros Proyectos
                                </h1>
                            </div>
                        </div>
                        <p className="text-lg text-muted-foreground mt-4 leading-relaxed">
                            Iniciativas para promover el diálogo intercultural, la incidencia política 
                            y la preservación del conocimiento ancestral.
                        </p>
                    </motion.div>

                    {/* Projects grid */}
                    <motion.div
                        variants={staggerContainer}
                        initial="hidden"
                        animate="visible"
                        className="mt-12 grid gap-6 lg:grid-cols-3"
                    >
                        {projects.map((project) => {
                            const Icon = project.icon;
                            const status = statusConfig[project.status];
                            const iconClass = colorClasses[project.color as keyof typeof colorClasses];

                            return (
                                <motion.div key={project.id} variants={fadeInUp}>
                                    <Card className="h-full group hover:border-forest/30 hover:shadow-xl transition-all duration-300">
                                        <CardHeader>
                                            <div className="flex items-start justify-between">
                                                <motion.div 
                                                    className={`p-3 rounded-2xl ${iconClass}`}
                                                    whileHover={{ scale: 1.1, rotate: 5 }}
                                                    transition={{ type: 'spring', stiffness: 400 }}
                                                >
                                                    <Icon className="h-6 w-6" />
                                                </motion.div>
                                                <Badge className={status.class}>
                                                    {status.label}
                                                </Badge>
                                            </div>
                                            <CardTitle className="mt-4 text-xl group-hover:text-forest transition-colors">
                                                {project.title}
                                            </CardTitle>
                                        </CardHeader>
                                        <CardContent>
                                            <CardDescription className="text-base leading-relaxed">
                                                {project.description}
                                            </CardDescription>
                                            
                                            {project.href && (
                                                <a 
                                                    href={project.href}
                                                    target="_blank"
                                                    rel="noopener noreferrer"
                                                    className="inline-block mt-6"
                                                >
                                                    <Button variant="outline" className="gap-2 hover:border-forest hover:text-forest">
                                                        Ver proyecto
                                                        <ExternalLink className="h-4 w-4" />
                                                    </Button>
                                                </a>
                                            )}
                                        </CardContent>
                                    </Card>
                                </motion.div>
                            );
                        })}
                    </motion.div>
                </div>
            </section>
        </PublicLayout>
    );
}
